
%% Copyright (C) 2001, James B. Rawlings and John G. Ekerdt
%%
%% This program is free software; you can redistribute it and/or
%% modify it under the terms of the GNU General Public License as
%% published by the Free Software Foundation; either version 2, or (at
%% your option) any later version.
%%
%% This program is distributed in the hope that it will be useful, but
%% WITHOUT ANY WARRANTY; without even the implied warranty of
%% MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
%% General Public License for more details.
%%
%% You should have received a copy of the GNU General Public License
%% along with this program; see the file COPYING.  If not, write to
%% the Free Software Foundation, 59 Temple Place - Suite 330, Boston,
%% MA 02111-1307, USA.


xa  = 0.75;
Rg  = 8.314;  % J/K mol
P   = 4*1.013e5;  % N/m^2
T   = 550; % K
ctf = P/(Rg*T)*1e-6; % mol/cm^3
Rp  = 0.45; % cm radius of catalyst particle
a   = Rp/3;
Nafin = 10; % mol/sec
NIf   = 10;
Ntf   = Nafin+NIf;
caf = ctf*0.5;
Qf  = Nafin/caf;
k1  = 2.25e5;  %cm^3/mol s
Da   = 0.008; % cm^2/s
rhob = 0.60; % g/cm^3 bed density
rhop = 0.68; % g/cm^3 particle density
epsb = 1 - rhob/rhop; % bed porosity, dimensionless

p.epsb = epsb;
p.a = a;
p.Nafin = Nafin;
p.Qf = Qf;
p.k1 = k1;
p.Da = Da;
p.caf = caf;
p.xa = xa;

%% collocation
npts = 25;
[R A B Q] = colloc(npts-2, 'left', 'right');
R = R*Rp;
A = A/Rp;
B = B/(Rp*Rp);
Q = Q*Rp;

p.R = R;
p.A = A;
p.B = B;

function retval = pellet(x, p)
  %% component A
  ca = x;
  r1      = p.k1.*ca.*ca;
  Ra      = - r1;
  retval = p.B*ca + 2*p.A*ca./p.R + Ra/p.Da;
  %% write over first and last residuals with BCs
  retval(1)  = p.A(1,:)*ca;
  retval(end,1) = p.caf - ca(end);
end%function

%% find the pellet profile at bed inlet
ca0 = logspace(log10(caf)-2,log10(caf),npts)';
tol = 1e-12;
opts = optimset ('TolFun', tol);
[ca,fval,info] = fsolve(@(x) pellet(x,p), ca0, opts);

info;

function res = bed(t, y, ydot, p)
  Naf = y(1);
  cpellet = y(2:end);
  Q  = p.Qf;
  p.caf = Naf/Q;
  %% calculate pellet residual and update
  %% total pellet reaction rate through dcadr
  pelletres = pellet(cpellet, p);
  dcadr = p.A(end,:)*cpellet;
  r1p   = p.Da/p.a*dcadr;
  RA    = -(1-p.epsb)*r1p;
  res(1) = ydot(1) - RA;
  res(2:length(y)) = pelletres;
  % matlab wants column vector
  res = res(:);
end%function

function [retval, isterminal, direction] = stop(t, y, ydot, p)
  Naf = y(1);
  convtest = p.xa - (1-Naf/p.Nafin);
  retval = convtest;
  isterminal = 1;
  direction = 0;
endfunction

%% march down the bed
nvs    = 100;
vfinal = 4e5;
vsteps = linspace (0,vfinal,nvs)';
vout   = vsteps;
y0     = [Nafin; ca];
ydot0    = zeros(length(y0),1);
res      = bed(0, y0, ydot0, p);
ydot0(1) = -res(1);

ymin = min(y0);
opts = odeset ('AbsTol', sqrt (eps), 'RelTol', 1e-10, ...
	       'Events', @(t, y, ydot) stop(t, y, ydot, p) );
[vout,y] = ode15i (@(t,y,ydot) bed(t,y,ydot,p), vsteps, y0, ydot0, opts);
nout = length(vout);
if ( nout == nvs )
  fprintf ('hey, did not reach final conversion, increase vfinal\n');
endif
xa = (Nafin-y(end,1))/Nafin;
Naf = y(:,1);
vplot = vout/1000.; %lit
VR = vplot(end);
tableex = [vplot, Naf];

%% pick out some good length locations for pellet profiles
rowsc = [1,nout];
colsc = [2:npts+1];
ca   = y(rowsc,colsc)';
table2 = [R, ca];
Naout = (1-xa)*Nafin;
Natop = (1-xa+0.10)*Nafin;
## dashedlines = [0,      Naout, VR, 0,     300,  Naout, VR, 2  ;
##                1.1*VR, Naout, VR, Natop, 400,  Naout, VR, 3  ];

%% Compare to the two approximations given in ch7,
%% Example 7.5, Figure 7.26

par.k       = k1;
par.Nafin   = Nafin;
par.T     = T;
par.rhop  = rhop;
par.rhob  = rhob;
par.Da    = Da;
par.Rp    = Rp;
par.Rg    = 82.06;
par.P     = 4;
par.n     = 2;
par.xa    = xa;
par.nvs  = nvs;
par.vfinal= vfinal;

## solve reactor with: eta = 1./Phi*( 1./tanh(3*Phi) - 1/(3*Phi) );
par.eta = (@(x) 1./x*( 1./tanh(3*x) - 1/(3*x) ));
[vap1, xap1] = pbrsolve(par);

## solve reactor with:  eta = 1./Phi;
par.eta = (@(x) 1./x);
[vap2, xap2] = pbrsolve(par);

vap1 = vap1/1000.;
VRap1 = vap1(end);

vap2 = vap2/1000.;
VRap2 = vap2(end);
tableap1 = [vap1 xap1];
tableap2 =  [vap2 xap2];
dashedlines = ...
[0,      Naout, VR, 0,     VRap1, 0,     VRap2, 0; ...
 1.1*VR, Naout, VR, Natop, VRap1, Natop, VRap2, Natop];

save fb2colloc.dat tableex tableap1 tableap2 dashedlines

if (~ strcmp (getenv ('OMIT_PLOTS'), 'true')) %% PLOTTING
%%plot the molar flow versus reactor volume and 75% conversion line
  plot(vplot, Naf, ...
       vap1, xap1, ...
       vap2, xap2, ...
       dashedlines(:,1), dashedlines(:,2), ...
       dashedlines(:,3), dashedlines(:,4), ...
       dashedlines(:,5), dashedlines(:,6), ...
       dashedlines(:,7), dashedlines(:,8))

axis ([0, 400, 2, 10])
%% TITLE
endif %% PLOTTING