
%% Copyright (C) 2001, James B. Rawlings and John G. Ekerdt
%%
%% This program is free software; you can redistribute it and/or
%% modify it under the terms of the GNU General Public License as
%% published by the Free Software Foundation; either version 2, or (at
%% your option) any later version.
%%
%% This program is distributed in the hope that it will be useful, but
%% WITHOUT ANY WARRANTY; without even the implied warranty of
%% MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
%% General Public License for more details.
%%
%% You should have received a copy of the GNU General Public License
%% along with this program; see the file COPYING.  If not, write to
%% the Free Software Foundation, 59 Temple Place - Suite 330, Boston,
%% MA 02111-1307, USA.
%
% This program solves the ethane pyrolysis text example with NO inhibitions.
%
% It was last edited 1/2/2002

%% Revised 7/24/2018

%% Cleaned up by jbr, 1/15/2022

%E in Joules, mass in grams, T in Kelvin, time in sec, volume in cm3

%%Components 1 = C2H6; 2 = NO; 3 = C2H5; 4 = HNO; 5 = H; 6 = C2H4; 7 = H2

A0 = [1e14, 1e12, 3e14, 3.4e12, 1e12, 1e13]';
Ea = [217600, 0, 165300, 28500, 0, 200800]';
R = 8.3144;  %(J/gmol-K)
R1 = 82.057;	%cc-atm/gmol-K
T = 1050;  %K
P = 1.0;     % atm
y1f  = 0.95;
y2f = 0.05;
C1f = y1f*P/(R1*T); %gmol/cm3
C2f = y2f*P/(R1*T); %gmol/cm3
Qf = 600.0;	%cc/sec
N1f = C1f*Qf;	%gmol/sec
N2f = C2f*Qf;
p = struct(); %Create structure to supply parameters to function
p.P = P;
p.T = T;
p.R1 = R1;
%% compute the rate constants k = A exp(-E/(RT)
tags = cellstr (['k1'; 'k_1'; 'k2'; 'k3'; 'k4'; 'k_4']);
for i = 1:length(tags)
  p.(tags{i}) = A0(i)*exp(-Ea(i)/(R*T));
end%for

function dNdv = rxrate(v, N, p)
  C =  N/sum(N)*p.P/(p.R1*p.T);
  r1 = p.k1*C(1)*C(2) - p.k_1*C(3)*C(4);
  r2 = p.k2*C(3);
  r3 = p.k3*C(1)*C(5);
  r4 = p.k4*C(2)*C(5) - p.k_4*C(4);
  dNdv = [-r1 - r3; ...
	  -r1 - r4; ...
	  r1 - r2 + r3; ...
	  r1 + r4; ...
	  r2 - r3 - r4; ...
	  r2; ...
	  r3];
end%function

N0 = [N1f, N2f, 0, 0, 0, 0, 0]';
v = linspace(0, 1500, 200)';
opts = odeset ('AbsTol', sqrt (eps), 'RelTol', sqrt (eps));
[tsolver, N] = ode15s(@(v,x) rxrate(v,x,p), v, N0, opts);
table = [v N];

save -ascii ethane_NO_1.dat table;

if (~ strcmp (getenv ('OMIT_PLOTS'), 'true')) %% PLOTTING
  ind = [1,2,6];  % components to plot: C2H6, NO, C2H4
  plot (v, N(:,ind));
  axis([0 1600 0 .007])
  %% TITLE
end%if %% PLOTTING