/*

Copyright (C) 2001, James B. Rawlings

This program is free software; you can redistribute it and/or
modify it under the terms of the GNU General Public License as
published by the Free Software Foundation; either version 2, or (at
your option) any later version.

This program is distributed in the hope that it will be useful, but
WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; see the file COPYING.  If not, write to
the Free Software Foundation, 59 Temple Place - Suite 330, Boston,
MA 02111-1307, USA.

*/

#include <octave/oct.h>
#include <octave/parse.h>
#include <octave/variables.h>
#include <octave/oct-map.h>
#include <octave/oct-rand.h>
#include <octave/quit.h>

DEFUN_DLD (stochsim, args, ,
	   "stochsim(x0, k, stoiT, tout)")

{
  octave_value retval;

  int nargin  (args.length ());

  if (nargin < 4 || nargin > 4)
    {
      print_usage ("stochsim");
      return retval;
    }

  const ColumnVector xin (args(0) . vector_value ());
  const ColumnVector k (args(1) . vector_value ());
  const Matrix stoiT (args(2) . matrix_value ());
  const ColumnVector tout (args(3) . vector_value ());

  // if (error_state)
  //   {
  //     error ("stochsim: invalid arguments");
  //     return retval;
  //   }

  const int nts  = tout.numel ();
  const double tfin = tout(nts-1);

  double time = tout(0);
  int iout = 0;

  const int nx = xin.numel ();

  if (nx != 3)
    {
      error ("stochsim: expecting X to be a 3-element vector");
      return retval;
    }

  double x0 = xin(0);
  double x1 = xin(1);
  double x2 = xin(2);

  Matrix xout (nx, nts, 0.0);

  double *pxout = xout.fortran_vec ();

  int pxout_offset = 0;

  double k0 = k(0);
  double k1 = k(1);
  double k2 = k(2);
  double k3 = k(3);
  double k4 = k(4);
  double k5 = k(5);

  while (time < tfin) 
    {
      // Allow this loop to be interrupted.
      OCTAVE_QUIT;

      double r0 = k0 * x0;
      double r1 = k1 * x1;
      double r2 = k2 * x0;
      double r3 = k3 * x0;
      double r4 = k4 * x2;
      double r5 = k5 * x1 * x2;

      double rtot = r0 + r1 + r2 + r3 + r4 + r5;

      // return if system has extinguished
      if (rtot == 0.0)
	{
	  while (iout < nts)
	    {
	      pxout[pxout_offset++] = x0;
	      pxout[pxout_offset++] = x1;
	      pxout[pxout_offset++] = x2;

	      iout++;
	    }
	  break;
	}

      // choose likely time of next reaction
      double timenew = time - log (octave::rand::scalar ()) / rtot;

      // choose which reaction (mth) is likely to occur
      double rnd = octave::rand::scalar ();
      double rcum = r0;
      int m = 0;
      double tmp = rnd*rtot;
      if (rcum <= tmp)
	{
	  ++m; rcum += r1;

	  if (rcum <= tmp)
	    {
	      ++m; rcum += r2;

	      if (rcum <= tmp)
		{
		  ++m; rcum += r3;

		  if (rcum <= tmp)
		    {
		      ++m; rcum += r4;

		      if (rcum <= tmp)
			++m; rcum += r5;
		    }
		}
	    }
	}


      double xnew0 = x0 + stoiT(0,m);
      double xnew1 = x1 + stoiT(1,m);
      double xnew2 = x2 + stoiT(2,m);

      // check if passing time of requested output
      while (tout(iout) <= timenew)
	{
	  pxout[pxout_offset++] = x0;
	  pxout[pxout_offset++] = x1;
	  pxout[pxout_offset++] = x2;

	  iout++;

	  if (iout >= nts)
	    break;
	}

      time = timenew;

      x0 = xnew0;
      x1 = xnew1;
      x2 = xnew2;
    }

  retval = xout;

  return retval;
}
