%% Copyright (C) 2001, James B. Rawlings and John G. Ekerdt
%%
%% This program is free software; you can redistribute it and/or
%% modify it under the terms of the GNU General Public License as
%% published by the Free Software Foundation; either version 2, or (at
%% your option) any later version.
%%
%% This program is distributed in the hope that it will be useful, but
%% WITHOUT ANY WARRANTY; without even the implied warranty of
%% MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
%% General Public License for more details.
%%
%% You should have received a copy of the GNU General Public License
%% along with this program; see the file COPYING.  If not, write to
%% the Free Software Foundation, 59 Temple Place - Suite 330, Boston,
%% MA 02111-1307, USA.
%%
%% Revised 7/24/2018
%%

%% add a stochastic simulation using Gillespie's algorithm
%%
%% example 1: A + B  --> C
%%            C      --> A + B
%%

nmolec = 400;
k1 = 1;
k2 = 1;
k(1) = k1/(nmolec);
k(2) = k2;
stoi = [-1 -1 1;
         1 1 -1];
[nrxs,nspec] = size(stoi);

clear x
x(1,1) = nmolec;
x(2,1) = 0.9*nmolec;
x(3,1) = 0*nmolec;

stoiT = stoi';
nsim = nmolec*4;

clear time
time = zeros(nsim+1,1);
time(1) = 0;
rand('seed', 0);

for n=1:nsim
    r(1) = k(1)*x(1,n)*x(2,n);
    r(2) = k(2)*x(3,n);
    rtot = sum(r);
    pp = rand(2,1);
    tau = -log(pp(1))/rtot;
    time(n+1) = time(n) + tau;

    %% determine which reaction (mth) is likely to occur
    m = sum (cumsum (r) <= pp(2)*rtot) + 1;
    x(:,n+1) = x(:,n) + stoiT(:,m);
endfor

%%
%% scale  so ca0=1;
%%
[ts,xs] = stairs(time, x');
ts = ts(:,1);
xscale = xs ./ xs(1,1);
%plot(time,xscale')
%%
%% make a deterministic comparison for nonlinear example 1.
%%
ca0 = x(1,1)/x(1,1);
cb0 = x(2,1)/x(1,1);
cc0 = x(3,1)/x(1,1);
timedet = [0; (logspace (-4,1,100)')];
k1det = k1;
k2det = k2;
%%
%% analytical if k2det = 0
%%
% delc = ca0-cb0;
% ca = delc ./ ( 1 - cb0/ca0 .* exp(-delc*k1det .* timedet));
% cb = ca - delc;
% cc = - ca + ( cc0 + ca0);

function xdot = f(t,x,p)
    ca = x(1);
    cb = ca - p.ca0 + p.cb0;
    cc = p.cc0 + p.ca0 - ca;
    xdot(1) = -p.k1det*ca*cb + p.k2det*cc;
endfunction

p = struct(); % Create structure to pass parameters to fsolve function
p.ca0 = ca0;
p.cb0 = cb0;
p.cc0 = cc0;
p.k1det = k1det;
p.k2det = k2det;

opts = odeset ('AbsTol', sqrt (eps), 'RelTol', sqrt (eps));
[tsolver, ca] = ode15s(@(t,x) f(t,x,p), time, ca0, opts);

cb = ca - ca0 + cb0;
cc = cc0 + ca0 - ca;
%plot(timedet, [ca,cb,cc])

table1 = [ts xscale];
table2 = [time ca cb cc];
save stochnon.dat table1 table2;

if (~ strcmp (getenv ('OMIT_PLOTS'), 'true')) %% PLOTTING
    plot (table1(:,1),table1(:,2:4), table2(:,1), table2(:,2:4));
    axis ([0, 5, 0, 1]);
    %% TITLE
endif %% PLOTTING