%% Copyright (C) 2001, James B. Rawlings and John G. Ekerdt
%%
%% This program is free software; you can redistribute it and/or
%% modify it under the terms of the GNU General Public License as
%% published by the Free Software Foundation; either version 2, or (at
%% your option) any later version.
%%
%% This program is distributed in the hope that it will be useful, but
%% WITHOUT ANY WARRANTY; without even the implied warranty of
%% MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
%% General Public License for more details.
%%
%% You should have received a copy of the GNU General Public License
%% along with this program; see the file COPYING.  If not, write to
%% the Free Software Foundation, 59 Temple Place - Suite 330, Boston,
%% MA 02111-1307, USA.

% This program solves part of ethane pyrolysis example that is in the
% text.  It plots the exact and the simple solutions
% It is titled ethane_comparison.m
%
% It was last edited 2/5/97
%%
%% Revised 7/24/2018

%E in Joules, mass in grams, T in Kelvin, time in sec, volume in cm3


Components_1 = [' C2H6 = 1,',' CH3 = 2,',' CH4 = 3,',' C2H5 = 4'];
Components_2 = [' H = 5,',' C2H4 = 6,', ' H2 = 7,',' H2O = 8'];

Ao = [1e17, 2e11, 3e14, 3.4e12, 1.6e13]';
Ea = [356000, 44000, 165000, 28000, 0]';
nu = [-1,2,0,0,0,0,0,0
      -1,-1,1,1,0,0,0,0
       0,0,0,-1,1,1,0,0
      -1,0,0,1,-1,0,1,0
       1,0,0,-1,-1,0,0,0];

R = 8.3144;  % Gas Constant (J/gmole-K)
p.R1 = 82.057; % Gas Constant (cc-atm/gmole-K)
p.T = 925; % Temp (K)
EXP = exp(-Ea/(R*p.T));
p.k = Ao.*EXP;
p.kp = (p.k(1)/(2*p.k(3)) + ((p.k(1)/(2*p.k(3)))^2 + ...
     ((p.k(1)*p.k(4))/(p.k(3)*p.k(5))))^0.5);
C1o = (50/760)/(82.057*p.T);  %gmole/cm3
C8o = (710/760)/(82.057*p.T);
Qf = 35.0;	%cc/sec
N1o = C1o*Qf;	%gmole/sec
N8o = C8o*Qf;
p.P = 1.0;	%atm
Initial = [N1o,0,0,0,0,0,0,N8o]';

function dNdv = rxrate(v,x,p)
    N1 = x(1);
    N2 = x(2);
    N3 = x(3);
    N4 = x(4);
    N5 = x(5);
    N6 = x(6);
    N7 = x(7);
    N8 = x(8);

    Ntot = N1 + N2 + N3 + N4 + N5 + N6 + N7 + N8;
    Ctot = p.P/(p.R1*p.T);
    C1 = (N1/Ntot)*Ctot;
    C2 = (N2/Ntot)*Ctot;
    C4 = (N4/Ntot)*Ctot;
    C5 = (N5/Ntot)*Ctot;

    r1 = p.k(1)*C1;
    r2 = p.k(2)*C1*C2;
    r3 = p.k(3)*C4;
    r4 = p.k(4)*C1*C5;
    r5 = p.k(5)*C4*C5;

    dNdv = zeros (8, 1);

    dNdv(1) = -r1 - r2 - r4 + r5;
    dNdv(2) = 2*r1 - r2;
    dNdv(3) = r2;
    dNdv(4) = r2 - r3 + r4 - r5;
    dNdv(5) = r3  - r4 - r5;
    dNdv(6) = r3;
    dNdv(7) = r4;
    dNdv(8) = 0;
endfunction

v = [0:0.5:100]';

opts = odeset ('AbsTol', sqrt (eps), 'RelTol', sqrt (eps));
[tsolver, solution] = ode15s(@(v,x) rxrate(v,x,p),v,Initial,opts);
answer = [v solution];

Initial_s = [N1o,0,0,N8o]';
function dNdv = rate_s(v,x,p)
    N1 = x(1);
    N6 = x(2);
    N7 = x(3);
    N8 = x(4);

    Ntot = N1 + N6 + N7 + N8;
    Ctot = p.P/(p.R1*p.T);
    C1 = (N1/Ntot)*Ctot;
    r = p.k(3)*p.kp*C1;

    dNdv = zeros (4, 1);

    dNdv(1) = -r;
    dNdv(2) = r;
    dNdv(3) = r;
    dNdv(4) = 0;
endfunction

[tsolver, solution_s] = ode15s(@(v,x) rate_s(v,x,p),v,Initial_s,opts);
answer_s = [v solution_s];

temp = [v, solution, solution_s];
save -ascii ethane_comparison.dat temp;

if (~ strcmp (getenv ('OMIT_PLOTS'), 'true')) %% PLOTTING
    plot (temp(:,1),[temp(:,2),temp(:,10),temp(:,7),temp(:,11)]);
    %% TITLE
endif %% PLOTTING