%% Copyright (C) 2001, James B. Rawlings and John G. Ekerdt
%%
%% This program is free software; you can redistribute it and/or
%% modify it under the terms of the GNU General Public License as
%% published by the Free Software Foundation; either version 2, or (at
%% your option) any later version.
%%
%% This program is distributed in the hope that it will be useful, but
%% WITHOUT ANY WARRANTY; without even the implied warranty of
%% MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
%% General Public License for more details.
%%
%% You should have received a copy of the GNU General Public License
%% along with this program; see the file COPYING.  If not, write to
%% the Free Software Foundation, 59 Temple Place - Suite 330, Boston,
%% MA 02111-1307, USA.
%%
%% Revised 8/13/2018

%
% multiplicity parameters
%
% units are kmol, min, kJ, K, m^3
%
%
p = struct();
p.k_m      = 0.001;
p.T_m      = 298;
p.E        = 8000;
p.c_Af     = 2;
p.C_p      = 4;
p.rho      = 1000;
p.C_ps     = p.rho*p.C_p;
p.T_f      = 298;
p.T_a      = p.T_f;
p.U        = 0;
p.DeltaH_R = -3e5;

function retval = st_st_cA(x, c_A, p)
    theta = x(1);
    T     = x(2);
    k     = p.k_m*exp(-p.E*(1/T - 1/p.T_m));
    retval(1) = p.c_Af - (1 + k*theta)*c_A;
    retval(2) = p.U*theta*(p.T_a - T) + p.C_ps*(p.T_f - T) -...
                    k*theta*c_A*p.DeltaH_R;
endfunction

x0 = [1; p.T_f];
nc_As = 250;
clear tmp_table;
c_Avect = linspace(0.999*p.c_Af, .003*p.c_Af, nc_As);

for i = 1: nc_As
    c_A = c_Avect(i);
    opts = optimset ('MaxFunEvals', 2000*numel (x0), ...
                   'MaxIter', 500*numel (x0));
    [x, fval, info] = fsolve(@(x) st_st_cA(x, c_A, p), x0, opts);
    theta = x(1);
    T     = x(2);
    conv  = (p.c_Af - c_A)/p.c_Af;

    %% compute the eigenvalues of the Jacobian
    k = p.k_m*exp(-p.E*(1/T - 1/p.T_m));
    Jac = [-1/theta - k,...
           -k*c_A*p.E/(T*T);
           -k*p.DeltaH_R/p.C_ps,...
           -p.U/p.C_ps - 1/theta - k*c_A*p.DeltaH_R/p.C_ps*p.E/(T*T)];

    lambda = eig(Jac);
    a = real(lambda(1));
    b = imag(lambda(1));
    c = real(lambda(2));
    d = imag(lambda(2));

    if (a >= c)
        lamrow = [a b c d];
    else
        lamrow = [c d a b];
    endif

    tmp_table(i,:) = [theta, T, conv, lamrow, info];
    x0 = x;
endfor

table = [tmp_table];
save -ascii st_st_eig.dat table;

if (~ strcmp (getenv ('OMIT_PLOTS'), 'true')) %% PLOTTING
    plot(table(:,2),[table(:,4),table(:,6)]);
    axis([280,460,-1,1]);

    figure()
    plot(table(:,3),[table(:,4),table(:,6)]);
    axis([0,1,-1,1]);
    %% TITLE
endif %% PLOTTING