%% Copyright (C) 2001, James B. Rawlings and John G. Ekerdt
%%
%% This program is free software; you can redistribute it and/or
%% modify it under the terms of the GNU General Public License as
%% published by the Free Software Foundation; either version 2, or (at
%% your option) any later version.
%%
%% This program is distributed in the hope that it will be useful, but
%% WITHOUT ANY WARRANTY; without even the implied warranty of
%% MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
%% General Public License for more details.
%%
%% You should have received a copy of the GNU General Public License
%% along with this program; see the file COPYING.  If not, write to
%% the Free Software Foundation, 59 Temple Place - Suite 330, Boston,
%% MA 02111-1307, USA.

%%
%% limit cycle parameters
%%
%% units are kmol, min, kJ, K, m^3
%%
%% 9/16/98, jbr
%%
%% Revised 8/13/2018

p = struct();
p.k_m      = 0.004;
p.T_m      = 298;
p.E        = 15000;
p.c_Af     = 2;
p.C_p      = 4;
p.rho      = 1000;
p.C_ps     = p.C_p*p.rho;
p.T_f      = 298;
p.T_a      = p.T_f;
p.DeltaH_R = -2.2e5;
p.U        = 340;
p.theta    = 35;
p.T_set    = 321.53;
p.c_set    = 0.48995;
p.T_fs     = p.T_f;
p.Kc       = 0;
p.gamma    = p.E/p.T_f;
p.B        = -p.DeltaH_R*p.c_Af*p.gamma/(p.C_ps*p.T_f);
p.beta     = p.U/p.C_ps*p.theta;
p.Da       = p.k_m*exp(-p.E*(1/p.T_f-1/p.T_m))*p.theta;
p.x2c      = (p.T_a-p.T_f)/p.T_f*p.gamma;

function retval = st_st_cA(x, c_A, p)
    theta = x(1);
    T     = x(2);
    k         = p.k_m*exp(-p.E*(1/T - 1/p.T_m));
    retval(1) = p.c_Af - (1 + k*theta)*c_A;
    retval(2) = p.U*theta*(p.T_a - T) + p.C_ps*(p.T_f - T) -...
                    k*theta*c_A*p.DeltaH_R;
endfunction

%%
%% building the lower branch;
%% use (theta,T) as dependent and
%% c_A as independent
%%
x0 = [1; p.T_f];
npts1 = 200;
xvect = logspace(log10(1e-4), log10(0.75), npts1);
c_Avect = (1-xvect)*p.c_Af;
clear tmp_table;

for i = 1:npts1
    c_A = c_Avect(i);
    opts = optimset ('MaxFunEvals', 2000*numel (x0), ...
                   'MaxIter', 500*numel (x0));
    [x, fval, info] = fsolve(@(x) st_st_cA(x, c_A, p), x0, opts);
    theta = x(1);
    T     = x(2);
    conv  = (p.c_Af - c_A)/p.c_Af;
    k = p.k_m*exp(-p.E*(1/T - 1/p.T_m));

    %% compute the eigenvalues of the Jacobian
    Jac = [-1/theta - k,...
           -k*c_A*p.E/(T*T);
           -k*p.DeltaH_R/p.C_ps,...
           -p.U/p.C_ps - 1/theta - k*c_A*p.DeltaH_R/p.C_ps*p.E/(T*T)];

    %% lambda = sort(real(eig(Jac)))';
    lambda = eig(Jac);
    a = real(lambda(1));
    b = imag(lambda(1));
    c = real(lambda(2));
    d = imag(lambda(2));

    if (a >= c)
        lamrow = [a b c d];
    else
        lamrow = [c d a b];
    endif

    tmp_table(i,:) = [theta, T, conv, lamrow, info];
    x0 = x;
endfor

table = [tmp_table];

%% stuff in some points near the extinction point to resolve
%% rapid change in the eigenvalues
npts2 = 200;
xnow = conv;
xvect = linspace(xnow, .95, npts2);
c_Avect = (1-xvect)*p.c_Af;
clear tmp_table;

for i = 1:npts2
    c_A = c_Avect(i);
    opts = optimset ('MaxFunEvals', 2000*numel (x0), ...
                   'MaxIter', 500*numel (x0));
    [x, fval, info] = fsolve(@(x) st_st_cA(x, c_A, p), x0, opts);
    theta = x(1);
    T     = x(2);
    conv  = (p.c_Af - c_A)/p.c_Af;
    k = p.k_m*exp(-p.E*(1/T - 1/p.T_m));

    %% compute the eigenvalues of the Jacobian
    Jac = [-1/theta - k,...
           -k*c_A*p.E/(T*T);
           -k*p.DeltaH_R/p.C_ps,...
           -p.U/p.C_ps - 1/theta - k*c_A*p.DeltaH_R/p.C_ps*p.E/(T*T)];

    lambda = eig(Jac);
    a = real(lambda(1));
    b = imag(lambda(1));
    c = real(lambda(2));
    d = imag(lambda(2));

    if (a >= c)
        lamrow = [a b c d];
    else
        lamrow = [c d a b];
    endif

    tmp_table(i,:) = [theta, T, conv, lamrow, info];
    x0 = x;
endfor

table = [table; tmp_table];

function retval = st_st_theta(x, theta, p)
    c_A   = x(1);
    T     = x(2);
    k         = p.k_m*exp(-p.E*(1/T - 1/p.T_m));
    retval(1) = p.c_Af - (1 + k*theta)*c_A;
    retval(2) = p.U*theta*(p.T_a - T) + p.C_ps*(p.T_f - T) -...
                    k*theta*c_A*p.DeltaH_R;
endfunction

%% after turning the corner on the upper branch; switch to (c_A,T) as
%% dependent and theta as independent load x with current solution
x0v = [c_A; T];
cortheta = theta;
npts3 = 200;
theta_vect = logspace(log10(cortheta), log10(20.7), npts3);
clear tmp_table;

for i = 1:npts3
    theta = theta_vect(i);
    [x, fval, info] = fsolve(@(x) st_st_theta(x, theta, p), x0, opts);
    c_A   = x(1);
    T     = x(2);
    conv  = (p.c_Af - c_A)/p.c_Af;
    k = p.k_m*exp(-p.E*(1/T - 1/p.T_m));

    %% compute the eigenvalues of the Jacobian
    Jac = [-1/theta - k,...
           -k*c_A*p.E/(T*T);
           -k*p.DeltaH_R/p.C_ps,...
           -p.U/p.C_ps - 1/theta - k*c_A*p.DeltaH_R/p.C_ps*p.E/(T*T)];

    lambda = eig(Jac);
    a = real(lambda(1));
    b = imag(lambda(1));
    c = real(lambda(2));
    d = imag(lambda(2));

    if (a >= c)
        lamrow = [a b c d];
    else
        lamrow = [c d a b];
    endif

    tmp_table(i,:) = [theta, T, conv, lamrow, info];
    x0 = x;
endfor

table = [table; tmp_table];

%% jam a bunch of points in interval theta=[20.7,20.8] to pick up pair
%% of conjugate eigenvalues branching from real axis theta as
%% independent load x with current solution
x0 = [c_A; T];
cortheta= theta;
npts4 = 100;
theta_vect = linspace(cortheta, 20.8, npts4);
clear tmp_table;

for i = 1:npts4
    theta = theta_vect(i);
    [x, fval, info] = fsolve(@(x) st_st_theta(x, theta, p), x0, opts);
    c_A   = x(1);
    T     = x(2);
    conv  = (p.c_Af - c_A)/p.c_Af;
    k = p.k_m*exp(-p.E*(1/T - 1/p.T_m));

    %% compute the eigenvalues of the Jacobian
    Jac = [-1/theta - k,...
           -k*c_A*p.E/(T*T);
           -k*p.DeltaH_R/p.C_ps,...
           -p.U/p.C_ps - 1/theta - k*c_A*p.DeltaH_R/p.C_ps*p.E/(T*T)];

    lambda = eig(Jac);
    a = real(lambda(1));
    b = imag(lambda(1));
    c = real(lambda(2));
    d = imag(lambda(2));

    if (a >= c)
        lamrow = [a b c d];
    else
        lamrow = [c d a b];
    endif

    tmp_table(i,:) = [theta, T, conv, lamrow, info];
    x0 = x;
endfor

table = [table; tmp_table];

%% finish out to large theta theta as independent load x with current
%% solution
x0 = [c_A; T];
cortheta = theta;
npts5 = 200;
theta_vect = logspace(log10(cortheta), log10(500), npts5);
clear tmp_table;

for i = 1: npts5
    theta = theta_vect(i);
    [x, fval, info] = fsolve(@(x) st_st_theta(x, theta, p), x0, opts);
    c_A   = x(1);
    T     = x(2);
    conv  = (p.c_Af - c_A)/p.c_Af;
    k = p.k_m*exp(-p.E*(1/T - 1/p.T_m));

    %% compute the eigenvalues of the Jacobian
    Jac = [-1/theta - k,...
           -k*c_A*p.E/(T*T);
           -k*p.DeltaH_R/p.C_ps,...
           -p.U/p.C_ps - 1/theta - k*c_A*p.DeltaH_R/p.C_ps*p.E/(T*T)];

    lambda = eig(Jac);
    a = real(lambda(1));
    b = imag(lambda(1));
    c = real(lambda(2));
    d = imag(lambda(2));

    if (a >= c)
        lamrow = [a b c d];
    else
        lamrow = [c d a b];
    endif

    tmp_table(i,:) = [theta, T, conv, lamrow, info];
    x0 = x;
endfor

table = [table; tmp_table];

%% for eigenvalue plot for s-part of curve, save rows 1:npts1+npts2
%%
%% for eigenvalue plot for limit cycle part of curve, save rows
%% 441:npts1+npts2+npts3+npts4+npts5
%%
%% Obviously the above row numbers change depending on how the points
%% were placed, so this part has to be redone if the problem parameters
%% change. I didn't see a better solution without going to a full
%% continuation approach, which seems like overkill for one or two
%% problems of this type. jbr 9/16/98

tmp = table(1:npts1 + npts2,:);
save -ascii st_st_osc_eigs.dat tmp;

if (~ strcmp (getenv ('OMIT_PLOTS'), 'true')) %% PLOTTING
    plot (tmp(:,3), tmp(:,[4,6]));
    axis ([0, 1, -200, 400]);

    figure()
    plot (tmp(:,3), tmp(:,[4,6]));
    axis ([0, 1, -1, 1]);
    %% TITLE
endif %% PLOTTING