%% Copyright (C) 2001, James B. Rawlings and John G. Ekerdt
%%
%% This program is free software; you can redistribute it and/or
%% modify it under the terms of the GNU General Public License as
%% published by the Free Software Foundation; either version 2, or (at
%% your option) any later version.
%%
%% This program is distributed in the hope that it will be useful, but
%% WITHOUT ANY WARRANTY; without even the implied warranty of
%% MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
%% General Public License for more details.
%%
%% You should have received a copy of the GNU General Public License
%% along with this program; see the file COPYING.  If not, write to
%% the Free Software Foundation, 59 Temple Place - Suite 330, Boston,
%% MA 02111-1307, USA.
%%
%% Revised 8/13/2018

%
% limit cycle parameters
%
% units are kmol, min, kJ, K, m^3
%

p = struct();
p.k_m      = 0.004;
p.T_m      = 298;
p.E        = 15000;
p.c_Af     = 2;
p.C_p      = 4;
p.rho      = 1000;
p.C_ps     = p.C_p*p.rho;
p.T_f      = 298;
p.T_a      = p.T_f;
p.DeltaH_R = -2.2e5;
p.U        = 340;
p.theta    = 30;
p.T_set    = 321.53;
p.c_set    = 0.48995;
p.T_fs     = p.T_f;
p.Kc       = 0;
p.gamma    = p.E/p.T_f;
p.B        = -p.DeltaH_R*p.c_Af*p.gamma/(p.C_ps*p.T_f);
p.beta     = p.U/p.C_ps*p.theta;
p.Da       = p.k_m*exp(-p.E*(1/p.T_f-1/p.T_m))*p.theta;
p.x2c      = (p.T_a-p.T_f)/p.T_f*p.gamma;

function retval = rhs(t, x, p)
    c_A   = x(1);
    T     = x(2);
    k     = p.k_m*exp(-p.E*(1/T - 1/p.T_m));
    T_f   = p.T_fs + p.Kc*(T - p.T_set);

    retval = zeros (2, 1);
    retval(1) = (p.c_Af - c_A)/p.theta - k*c_A;
    retval(2) = p.U/p.C_ps*(p.T_a - T) + (p.T_f - T)/p.theta -...
                    k*c_A*p.DeltaH_R/p.C_ps;
endfunction

x0 = [p.c_Af; p.T_f];
tfinal = 10*p.theta;
ntimes = 500;
tout   = linspace(0, tfinal, ntimes);

opts = odeset ('AbsTol', sqrt (eps), 'RelTol', sqrt (eps));
[tsolver, x] = ode15s(@(t, x) rhs(t, x, p), tout, x0, opts);

u = (x(:,2) - p.T_set)*p.Kc + p.T_fs;
conv = (p.c_Af - x(:,1))/p.c_Af;
table = [tout' x conv u];
save -ascii dyn_osc_hopf1.dat table;

if (~ strcmp (getenv ('OMIT_PLOTS'), 'true')) %% PLOTTING
    subplot(2,1,1);
    plot (table(:,1), table(:,4));
    %% TITLE dyn_osc_hopf1

    subplot(2,1,2);
    plot (table(:,1), table(:,3));
    %% TITLE dyn_osc_hopf1
endif %% PLOTTING