%% Copyright (C) 2001, James B. Rawlings and John G. Ekerdt
%%
%% This program is free software; you can redistribute it and/or
%% modify it under the terms of the GNU General Public License as
%% published by the Free Software Foundation; either version 2, or (at
%% your option) any later version.
%%
%% This program is distributed in the hope that it will be useful, but
%% WITHOUT ANY WARRANTY; without even the implied warranty of
%% MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
%% General Public License for more details.
%%
%% You should have received a copy of the GNU General Public License
%% along with this program; see the file COPYING.  If not, write to
%% the Free Software Foundation, 59 Temple Place - Suite 330, Boston,
%% MA 02111-1307, USA.
%%
%% Revised 8/13/2018

%
% limit cycle parameters
%
% units are kmol, min, kJ, K, m^3
%

p = struct();
p.k_m      = 0.004;
p.T_m      = 298;
p.E        = 15000;
p.c_Af     = 2;
p.C_p      = 4;
p.rho      = 1000;
p.C_ps     = p.C_p*p.rho;
p.T_f      = 298;
p.T_a      = p.T_f;
p.DeltaH_R = -2.2e5;
p.U        = 340;
%p.theta    = 72.3308;
p.theta    = 73.1;
p.T_set    = 321.53;
p.c_set    = 0.48995;
p.T_fs     = p.T_f;
p.Kc       = 0;
p.gamma    = p.E/p.T_f;
p.B        = -p.DeltaH_R*p.c_Af*p.gamma/(p.C_ps*p.T_f);
p.beta     = p.U/p.C_ps*p.theta;
p.Da       = p.k_m*exp(-p.E*(1/p.T_f-1/p.T_m))*p.theta;
p.x2c      = (p.T_a-p.T_f)/p.T_f*p.gamma;

function retval = rhs(t, x, p)
    c_A   = x(1);
    T     = x(2);
    k     = p.k_m*exp(-p.E*(1/T - 1/p.T_m));
    T_f   = p.T_fs + p.Kc*(T - p.T_set);

    retval = zeros (2, 1);
    retval(1) = (p.c_Af - c_A)/p.theta - k*c_A;
    retval(2) = p.U/p.C_ps*(p.T_a - T) + (p.T_f - T)/p.theta -...
                    k*c_A*p.DeltaH_R/p.C_ps;
endfunction

%% find the stable limit cycle
x0 = [(1-0.8)*p.c_Af 310];
tfinal = 40*p.theta;
ntimes = 3;
tout  = linspace(0, tfinal, ntimes);
opts = odeset('AbsTol', sqrt (eps), 'RelTol', sqrt (eps));
[tsolver, x] = ode15s(@(t, x) rhs(t, x, p), tout, x0, opts);

%% now go around the limit cycle once with many time points
x0 = x(end,:);
ntimes = 200;
tfinal = 3*p.theta;
tout = linspace(0, tfinal, ntimes);
opts = odeset('AbsTol', sqrt (eps), 'RelTol', sqrt (eps));
[tsolver, x] = ode15s(@(t, x) rhs(t, x, p), tout, x0, opts);
u = (x(:,2) - p.T_set)*p.Kc + p.T_fs;
conv = (p.c_Af - x(:,1))/p.c_Af;
stablim = [tout' x conv u];

%% reverse time and stabilize the unstable limit cycle starting in the
%% interior

function retval = reverserhs(t, x, p)
    c_A   = x(1);
    T     = x(2);
    k     = p.k_m*exp(-p.E*(1/T - 1/p.T_m));
    T_f   = p.T_fs + p.Kc*(T - p.T_set);
    retval = zeros (2, 1);
    retval(1) = -((p.c_Af - c_A)/p.theta - k*c_A);
    retval(2) = -(p.U/p.C_ps*(p.T_a - T) + (p.T_f - T)/p.theta -...
                    k*c_A*p.DeltaH_R/p.C_ps);
endfunction

%% find the unstable limit cycle
%% 73.4957904330517 305.839679413557 0.516502467355002 -0.00139021373543844 0.0329582458327707 -0.00139021373543844 -0.0329582458327707 1
%% stable steady state is about 0.5165, 305.83

x0=[(1-0.7)*p.c_Af; 305];
tfinal = 40*p.theta;
ntimes = 3;
tout   = linspace(0, tfinal, ntimes);
opts = odeset('AbsTol', sqrt (eps), 'RelTol', sqrt (eps));
[tsolver, x] = ode15s(@(t, x) reverserhs(t, x, p), tout, x0, opts);
%% now go around the unstable limit cycle once with many time points
x0=x(end,:);
tfinal = 3*p.theta;
ntimes = 200;
tout   = linspace(0, tfinal, ntimes);
opts = odeset('AbsTol', sqrt (eps), 'RelTol', sqrt (eps));
[tsolver, x] = ode15s(@(t, x) rhs(t, x, p), tout, x0, opts);

u = (x(:,2) - p.T_set)*p.Kc + p.T_fs;
conv = (p.c_Af - x(:,1))/p.c_Af;
unstablim = [tout' x conv u];

table = [stablim unstablim];
save -ascii phase_portrait_2.dat table;
size(stablim)
size(unstablim)
disp(max(table(:,3)))
disp(max(table(:,4)))
disp(max(table(:,8)))
disp(max(table(:,9)))
if (~ strcmp (getenv ('OMIT_PLOTS'), 'true')) %% PLOTTING
    plot (table(:,3),table(:,4),table(:,8),table(:,9));
    %% TITLE
endif %% PLOTTING