%% Copyright (C) 2001, James B. Rawlings and John G. Ekerdt
%%
%% This program is free software; you can redistribute it and/or
%% modify it under the terms of the GNU General Public License as
%% published by the Free Software Foundation; either version 2, or (at
%% your option) any later version.
%%
%% This program is distributed in the hope that it will be useful, but
%% WITHOUT ANY WARRANTY; without even the implied warranty of
%% MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
%% General Public License for more details.
%%
%% You should have received a copy of the GNU General Public License
%% along with this program; see the file COPYING.  If not, write to
%% the Free Software Foundation, 59 Temple Place - Suite 330, Boston,
%% MA 02111-1307, USA.

%%
%% jbr modifications to run on telemark; seems to have some parameters
%% set awfully high. Probably works on hazel with some better precision
%% than telemark.
%%
%% 6/4/00
%%
%% streamlined code, jbr, 12/29/2017
%%
%% Revised 8/14/2018

%% The pellet problem - Explanation of the structure of this program
%% This program computes the effectiveness factors for different thiele
%% moduli , for any order of the reaction and for the cases of
%% slab,cylinder and sphere(q = 0,1,2).

%% The orders can be can be categorised into order = 1 ,order<1 and
%% order > 1.

%% For order < 1 , the problem is solved in two stages
%% Stage 1.
%% The same intitial value problem approach as explained abouve is used,
%% but  here we march upto t = large no.This would yield all values of
%% thiele  modulus for which the concentration at the centre is positive
%% Stage 2
%% Here the discontinuity in the concentration profile is
%% accomodated. We slightly modify the above initial value problem
%% approach. In this we start at a radius t = start, with the initial
%% conditions as c0 = 0 and a very small derivative, then we march forward
%% upto t = start+tsteps. The profile is rescaled to satisfy the outer
%% boundary condition c((q+1)*(start+tsteps) = 1). Unlike the previous
%% case, the values of thiele modulus decreases for increase in
%% tsteps.The required solution is [Stage1;Stage2]

%% Limitations:

%% At extremely high thiele values (i.e >10000) for  order >1, the k and
%% c0 values chosen here will have to be changed suitably so that the
%% sensitivity of concentration w.r.t tsteps, is within the limit of
%% ode solver tolerances.
%% If one needs extremely high thiele modulus for order < 1, then ,the
%% initial derivative needs to be reduced and also the tsteps in stage 2
%% need to be very small.
%% Order  = 1,q = 1 will not be able to yield very high thiele values. This
%% is because the thiele modulus is only proportional to tsteps, but
%% overflows occur at high tsteps.


%% This derivative program cvsrnsmall.m calculates the concentration
%% profile for n<1 to show that an inner sphere of the pellet has
%% exactly zero concentration of A. That is a result of the order of the
%% reaction being less than one.


%% Choose the orders n, geometry (q = 0,1,2 for slab,cylinder,sphere) and
%% the thiele modulus upto which you require eta values.(If you do not
%% have an order = 1, q = 1 case then you can choose values of thiele
%% modulus to be as high as 1000, else you will have to have thiele to
%% be not greater than 100.

p = struct();
p.order = 0;
p.q = 2;

%%
%% The thiele values are chosen to so that for given order and geometry,
%% first one has nonzero center concentration, second one has zero
%% concentration right at center, and remaining ones have zero
%% conentration at finite radius.
%%
thielevec = [0.4; 0.57735; 0.8; 10];
%%
%% These are constants chosen arbitrarily
%% k     rate constant/diffusivity
%% c0    initial concentration at the centre
%% der   initial derivative for the case of dry regions in order <1
%% start the radius from which we start to march in the case of order<1 dry

p.k = 2;
c0 = 0.01;
der = 1e-8;
start = 1;
tab = [];

%% Stopping Critierion

function [stp, isterminal, direction] = g(t, x, thiele, p)
    if x(1) == 0
        pelthiele = Inf;
    else
        pelthiele = sqrt((p.order + 1)/2*x(1)^(p.order - 1)*p.k)...
                        *t/(p.q+1);
    endif

    %% stop at each desired thiele modulus
    stp = pelthiele - thiele;
    isterminal = 1;
    direction = 0;
endfunction

%% ODE's

function xdot = f(t, x, p)
    xdot = zeros(2, 1);
    xdot(1) = x(2);

    if t == 0
        xdot(2) = p.k*x(1)^p.order;
    else
        xdot(2) = -p.q/t*x(2) + p.k*x(1)^p.order;
    endif
endfunction

nt = length(thielevec);
table(1:nt) = {[]};
tout = [0; logspace(-2,5,400)'];
t0 = start;
nfine = 200;
ncoarse = 100;
fine = logspace(log10(start), log10(start + 1), nfine)';
coarse = logspace(log10(start + 2), log10(40), ncoarse)';
tout2 = [fine; coarse(2:ncoarse)];

%% Main program

for i = 1:nt
    thiele = thielevec(i);

    if i <= 2
        %% Stage 1
        x0 = [c0; 0];
        tpts = tout;
    else
        %% Stage 2
        x0 = [0; der];
        tpts = tout2;
    endif

    opts = odeset('Events', @(t, x) g(t, x, thiele, p),...
                  'AbsTol', sqrt(eps), 'RelTol', sqrt(eps));
    [tsteps, x] = ode15s(@(t, x) f(t, x, p), tpts, x0, opts);
    tabletmp = [tsteps./tsteps(end)*(p.q + 1), x(:,1)./x(end,1)];
    table(i) = {tabletmp};
endfor

save cvsrnsmall.dat table

if (~ strcmp (getenv ('OMIT_PLOTS'), 'true')) %% PLOTTING
    hold on;

    for i = 1:nt
        plot (table{i}(:,1), table{i}(:,2));
    endfor

    hold off;
    %% TITLE
endif %% PLOTTING