%% Copyright (C) 2001, James B. Rawlings and John G. Ekerdt
%%
%% This program is free software; you can redistribute it and/or
%% modify it under the terms of the GNU General Public License as
%% published by the Free Software Foundation; either version 2, or (at
%% your option) any later version.
%%
%% This program is distributed in the hope that it will be useful, but
%% WITHOUT ANY WARRANTY; without even the implied warranty of
%% MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
%% General Public License for more details.
%%
%% You should have received a copy of the GNU General Public License
%% along with this program; see the file COPYING.  If not, write to
%% the Free Software Foundation, 59 Temple Place - Suite 330, Boston,
%% MA 02111-1307, USA.

%%
%% added log transformation in pellet
%% jbr
%% 12/13/03
%% added homotopy to find inlet condition
%% jbr
%% 1/16/12

%% Removed gl0bal, jbr, 7/14/2019
%% Noticed cpu time went from 29.4 to 44.5 when changing from gl0bal to parameter struct
%%

Rg  = 8.314;  % J/K mol
Mair= 28.96; % g/mol, mol weight air
%Pin   = 1.5*1.013e5;  % N/m^2
Pin   = 2.0*1.013e5;  % N/m^2
Pext  = 1.013e5; % minimum allowed exit pressure
Tin   = 550; % K
%Tin   = 570; % K
Rt  = 10/2; %cm, tube radius
At  = pi*Rt*Rt; % cm^2, tube cross-sectional area
Ta  = 325; % K, ambient temperature
U    = 5.5e-3; % cal/(cm^2 K s), heat transfer coefficient
delH1 = (-67.63e3); % cal/mol CO;
delH2 = (-460.4e3); % cal/mol C_3H_6;
Cp  = 0.25; % cal/(g K), heat capacity of air
vis = 0.028e-2; % g/(cm s), viscosity of air
bt  = 1; % bed to tube area ratio
%bt  = 4; % bed to tube area ratio
%u   = 500/bt;  % cm/sec, feed gas velocity entering bed
u  = 75/bt;  % cm/sec, feed gas velocity entering bed
Ac  = bt*At;  % cm^2 area of bed, 4 x area of tube
Qin = u*Ac;  % cm^3/sec, feed volumetric flowrate
P   = Pin;
T   = Tin;
Q   = Qin;
cfin= P/(Rg*T)*1e-6; % mol/cm^3
Rp  = 0.175; % cm radius of catalyst particle
a   = Rp/3;
rhob = 0.51; % g/cm^3 bed density
rhop = 0.68; % g/cm^3 particle density
epsb = 1 - rhob/rhop; % bed porosity, dimensionless
%epsb = 0.4;
xc  = 0.996;
%xc  = 0.97;
%xc  = 1.0;

cafin = cfin*0.02;
cbfin = cfin*0.03;
ccfin = cfin*5e-4;

Nafin = Q*cafin;
Nbfin = Q*cbfin;
Ncfin = Q*ccfin;
Nfin  = [Nafin; Nbfin; Ncfin];
Ntfin = Q*cfin;
massf = Ntfin*Mair; % mass flowrate, g/s, remains constant
Cptot = massf*Cp;

alpha = 1;
%alpha = 2;
k100  = alpha*6.802e16*2.6e6*80/100*0.05/100;  %mol/cm^3 s
k200  = alpha*1.416e18*2.6e6*80/100*0.05/100;  %mol/cm^3 s
k10   = k100;
k20   = k200;
E1   = 13108; %K
E2   = 15109; %K
Ka0  = 8.099e6; % cm^3/mol
Kc0  = 2.579e8; % cm^3/mol
Ea   = - 409; %K
Ec   = 191; %K
Da   = 0.0487; % cm^2/s
Db   = 0.0469; % cm^2/s
Dc   = 0.0487; % cm^2/s
kma  = 0.4*9.76;   % cm/s
kmb  = 0.4*10.18;  % cm/s
kmc  = 0.4*9.76;   % cm/s

%% calculate initial reaction rates
k1      = k10*exp(-E1/Tin);
k2      = k20*exp(-E2/Tin);
Ka      = Ka0*exp(-Ea/Tin);
Kc      = Kc0*exp(-Ec/Tin);
den     = (1+Ka*cafin+Kc*ccfin)*(1+Ka*cafin+Kc*ccfin);
r1      = k1*cafin*cbfin/den;
r2      = k2*ccfin*cbfin/den;

%% adiabatic temperature rise and
%% Ucrit to achieve a zero intial temperature derivative
adrise = -(cafin*delH1+ccfin*delH2)/(Cp*cfin*Mair)
Ucrit  = (r1*delH1 + r2*delH2)/(2/Rt*(Ta-T))

%% collocation
npts = 40;
[R A B Q] = colloc(npts-2, 'left', 'right');
R = R*Rp;
A = A/Rp;
B = B/(Rp*Rp);
Q = Q*Rp;

p.npts = npts; p.A = A; p.B = B; p.R = R;
p.k10 = k10; p.k20 = k20; p.E1 = E1; p.E2 = E2;
p.Ea = Ea; p.Ec = Ec; p.Ka0 = Ka0; p.Kc0 = Kc0;
p.Da = Da; p.Db = Db; p.Dc = Dc;
p.kma = kma; p.kmb = kmb; p.kmc = kmc;
p.T = T;

p.Nafin = Nafin; p.Ncfin = Ncfin; p.Ntfin = Ntfin;
p.Pin = Pin; p.Tin = Tin;
p.epsb = epsb; p.a = a; p.Qin = Qin; p.Cptot = Cptot; p.U = U; p.Ta = Ta;
p.Rt = Rt; p.delH1 = delH1; p.delH2 = delH2;
p.Rp = Rp; p.vis = vis; p.massf = massf; p.Ac = Ac;
p.xc = xc; p.Pext = Pext;

function [resid, rate] = pelletfull(x, p)
  za = x(1:p.npts);
  zb = x(p.npts+1:2*p.npts);
  zc = x(2*p.npts+1:3*p.npts);
  ca = exp(za);
  cb = exp(zb);
  cc = exp(zc);

  k1      = p.k10*exp(-p.E1/p.T);
  k2      = p.k20*exp(-p.E2/p.T);
  Ka      = p.Ka0*exp(-p.Ea/p.T);
  Kc      = p.Kc0*exp(-p.Ec/p.T);
  den     = (1+Ka*ca+Kc*cc).^2;
  r1      = k1.*ca.*cb./den;
  r2      = k2*cc.*cb./den;
  Ra      = - r1;
  Rb      = - 1/2*r1  - 9/2*r2;
  Rc      = - r2;
  %% component A
  ip = 1;
  resid(ip:ip+p.npts-1) = p.B*za + p.A*za.*(p.A*za + 2./p.R) + Ra./(p.Da*ca);
  resid(ip) = p.A(1,:)*za;
  resid(ip+p.npts-1) = p.Da*p.A(end,:)*za - p.kma*(p.caf/ca(end) - 1);

  %% component B
  ip = p.npts+1;
  resid(ip:ip+p.npts-1) = p.B*zb + p.A*zb.*(p.A*zb + 2./p.R) + Rb./(p.Db*cb);
  resid(ip) = p.A(1,:)*zb;
  resid(ip+p.npts-1) =  p.Db*p.A(end,:)*zb - p.kmb*(p.cbf/cb(end) - 1);

  %% component C
  ip = 2*p.npts+1;
  resid(ip:ip+p.npts-1) = p.B*zc + p.A*zc.*(p.A*zc + 2./p.R) + Rc./(p.Dc*cc);
  resid(ip)  = p.A(1,:)*zc;
  resid(ip+p.npts-1) = p.Dc*p.A(end,:)*zc - p.kmc*(p.ccf/cc(end) - 1);

  %% pellet surface derivatives (for production rates in bed function)
  rate = [p.A(end,:)*ca;
	  p.A(end,:)*cb;
	  p.A(end,:)*cc];
end%function

function retval = pellet(x, p)
  [pelletresid, pelletrate] = pelletfull(x, p);
  retval = pelletresid;
end%function

%% find the pellet profile at tube inlet
caf = cafin; cbf=cbfin; ccf=ccfin;
p.caf = caf; p.cbf = cbf; p.ccf = ccf;
%% try a homotopy on the rate constant
kstep = 0.1; ksuc = 0; ktry=0; kmin = 1e-3;
z0 = log([caf*ones(npts,1); cbf*ones(npts,1); ccf*ones(npts,1)]);
while (ksuc < 1.0)
  ktry = min( 1.0, ktry+kstep);
  k10 = ktry*k100;
  k20 = ktry*k200;
  p.k10 = k10; p.k20 = k20;
  [z,fval,info] = fsolve(@(x) pellet(x,p), z0);
  fsolve_failed = info <= 0;
  if (fsolve_failed)
    ktry = ksuc;
    kstep = kstep/2;
    if (kstep < kmin)
      kstep, ksuc
      error ('cannot continue homotopy past current ksuc')
    endif
  else
    ksuc = ktry;
    z0 = z;
  end%if
end%while

za = z(1:npts);
zb = z(npts+1:2*npts);
zc = z(2*npts+1:3*npts);

ca = exp(za);
cb = exp(zb);
cc = exp(zc);

function res = bed(t, y, ydot, p)
  Naf = y(1);
  Nbf = y(2);
  Ncf = y(3);
  T   = y(4);
  P   = y(5);
  zpellet = y(6:length(y));
  %% update parameter struct before calling pellet
  Ntf  = p.Ntfin -1/2*(p.Nafin - Naf) + 1/2*(p.Ncfin - Ncf);
  Q  = p.Qin * p.Pin/P*T/p.Tin*Ntf/p.Ntfin;
  caf = Naf/Q;
  cbf = Nbf/Q;
  ccf = Ncf/Q;
  %% update the fluid parameters and call pellet
  p.caf = caf; p.cbf = cbf; p.ccf = ccf; p.Q = Q; p.T = T;

  %% calculate pellet residual and update
  %% total pellet reaction rate through dcadr dcbdr dccdr
  [pelletres, pelletrate] = pelletfull(zpellet, p);
  dcadr = pelletrate(1); dcbdr=pelletrate(2); dccdr = pelletrate(3);
  r1p   = p.Da/p.a*dcadr;
  r2p   = p.Dc/p.a*dccdr;

  Rj    = -(1-p.epsb)/p.a*[ p.Da*dcadr; p.Db*dcbdr; p.Dc*dccdr];
  dT    = ( -(p.delH1*r1p + p.delH2*r2p) + 2/p.Rt*p.U*(p.Ta-T) ) / p.Cptot;
%  dT    = 0;
  dP    = - (1-p.epsb)*Q / ( 2*p.Rp*p.epsb^3.*p.Ac^2. ) * ...
          ( 150*(1-p.epsb)*p.vis/(2*p.Rp) + 7/4*p.massf/p.Ac );
%  dP = 0;
  rhs = [Rj; dT; dP];
  res(1:5,1) = ydot(1:5) - rhs;
  res(6:length(y)) = pelletres;
end%function

function [retval, isterminal, direction] = stop(t, y, ydot, p)
  Ncf = y(3);
  P   = y(5);
  convtest = p.xc - (1-Ncf/p.Ncfin);
  presstest = P - p.Pext;
  retval = min(convtest, presstest);
  isterminal = 1;
  direction = 0;
end%function

%% march down the bed
nvs    = 201;
%vfinal = 5000;
%vfinal = 4681;
vfinal = 2000;
vsteps = linspace (0,vfinal,nvs)';
vout   = vsteps;
y0     = [Nfin; Tin; Pin; z];
ydot0    = zeros(length(y0),1);
res      = bed(0, y0, ydot0, p);
ydot0(1:5)    = -res(1:5);

ymin = min(y0);
opts = odeset ('RelTol', 1e-7, 'AbsTol', 1e-7, 'Events', @(t,y,ydot) stop(t,y,ydot,p));
t0=cputime();
[vout,y] = ode15i (@(t,y,ydot) bed(t,y,ydot,p), vsteps, y0, ydot0, opts);
cpu = cputime()-t0
nout = length(vout);
if ( nout == nvs )
  fprintf ('hey, did not reach final conversion, increase vfinal\n');
endif
xa = (Nafin-y(end,1))/Nafin
xb = (Nbfin-y(end,2))/Nbfin
xc = (Ncfin-y(end,3))/Ncfin
P=y(:,5);
T=y(:,4);
ctot = P./(Rg*T)*1e-6;
Nt = Ntfin - 1/2*(Nafin-y(:,1))+1/2*(Ncfin-y(:,3));
Q = Nt./ctot;
caf = y(:,1)./Q;
cbf = y(:,2)./Q;
ccf = y(:,3)./Q;
Patm = P/1.013e5;
table1 = [vout, caf, cbf, ccf, T, Patm];

%% pick out some good length locations for pellet profiles
nrows = 5;
%rowsp = ceil(linspace(1,nout,nrows));
rowsp = [1,5,10,50,90,100,nout]
vout(rowsp)
cols = [6:npts+5];
ca   = exp(y(rowsp,cols)');
cols = [npts+6:2*npts+5];
cb   = exp(y(rowsp,cols)');
cols = [2*npts+6:3*npts+5];
cc   = exp(y(rowsp,cols)');
table2 = [R, ca, cb, cc];

save multicolloc_log.dat table1 table2

if (~ strcmp (getenv ('OMIT_PLOTS'), 'true')) %% PLOTTING
subplot (2, 2, 1);
semilogy (table1(:,1), table1(:,2:4));
%% TITLE multicolloc_log_1

subplot (2, 2, 2);
plot (table1(:,1), table1(:,5));
%% TITLE multicolloc_log_2

subplot (2, 2, 4);
plot (table1(:,1), table1(:,6));
%% TITLE multicolloc_log_2

subplot (2, 2, 3);
semilogy (table2(:,1), table2(:,2:8));
%% TITLE multicolloc_log_3
endif %% PLOTTING