%% Copyright (C) 2001, James B. Rawlings and John G. Ekerdt
%%
%% This program is free software; you can redistribute it and/or
%% modify it under the terms of the GNU General Public License as
%% published by the Free Software Foundation; either version 2, or (at
%% your option) any later version.
%%
%% This program is distributed in the hope that it will be useful, but
%% WITHOUT ANY WARRANTY; without even the implied warranty of
%% MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
%% General Public License for more details.
%%
%% You should have received a copy of the GNU General Public License
%% along with this program; see the file COPYING.  If not, write to
%% the Free Software Foundation, 59 Temple Place - Suite 330, Boston,
%% MA 02111-1307, USA.

%%
%% jbr modifications to run on telemark; seems to have some parameters
%% set awfully high. Probably works on hazel with some better precision
%% than telemark.
%%
%% 6/4/00
%%
%% Revised 8/14/2018

%% The pellet problem - Explanation of the structure of this program
%% This program computes the effectiveness factors for different thiele
%% moduli , for any order of the reaction and for the cases of
%% slab,cylinder and sphere(q = 0,1,2).

%% The orders can be can be categorised into order=1 ,order<1 and
%% order>1.

%% For order>1, the problem is set up as an initial value problem. We
%% start from the centre of the pellet (t = 0) with the inner boundary condition
%% (dc/dr = 0) and some c0 = constant, and start marching forward in the
%% radial direction upto t = tsteps. The profile is then rescaled to satisfy
%% the outer boundary condition (c((q+1)*tsteps) = 1). For different
%% values of tsteps we have different thiele modulus. We use the ODE
%% solver with stopping condition
%% (ex.. thiele modulus = 100). The solver marches forward until the
%% stopping condition is satisfied.(The stopping condition helps to
%% avoid the problem of blowing up)

%% For order = 1, q = 0 or 2, the analytical solutions have been used
%% For order = 1, q = 1 the solution is computed just like the case of
%% order>1 (Analytical solution for this is actually available, but it
%% involves Modified Bessels function which limits the value of thiele
%% modulus that can be attained.

%% For order<1 , the problem is solved in two stages
%% Stage 1.
%% The same intitial value problem approach as explained abouve is used
%% , but  here we march upto t=large no.This would yield all values of
%% thiele  modulus for which the concentration at the centre is positive
%% Stage 2
%% Here the discontinuity in the concentration profile is
%% accomodated. We slightly modify the above initial value problem
%% approach. In this we start at a radius t=start, with the initial
%% conditions as c0 = 0 and a very small derivative, then we march forward
%% upto t = start+tsteps. The profile is rescaled to satisfy the outer
%% boundary condition c((q + 1)*(start + tsteps) = 1). Unlike the previous
%% case, the values of thiele modulus decreases for increase in
%% tsteps.The required solution is [Stage1; Stage2]

%% Limitations:

%% At extremely high thiele values (i.e > 10000) for  order > 1, the k and
%% c0 values chosen here will have to be changed suitably so that the
%% sensitivity of concentration w.r.t tsteps, is within the limit of
%% the ODE solver tolerances.
%% If one needs extremely high thiele modulus for order < 1, then ,the
%% initial derivative needs to be reduced and also the tsteps in stage 2
%% need to be very small.
%% Order = 1, q = 1 will not be able to yield very high thiele values. This
%% is because the thiele modulus is only proportional to tsteps, but
%% overflows occur at high tsteps.

%% Choose the orders , geometry (q = 0,1,2 for slab,cylinder,sphere) and
%% the thiele modulus upto which you require eta values.(If you do not
%% have an order = 1, q = 1 case then you can choose values of thiele
%% modulus to be as high as 1000, else you will have to have thiele to
%% be not greater than 100.

ordervec = [1; 2; 5; 10];
p = struct();
p.q = 2;
p.thiele = 10;

%% These are constants chosen arbitrarily
%% k     rate constant/diffusivity
%% c0    initial concentration at the centre
%% der   initial derivative for the case of dry regions in order <1
%% start the radius from which we start to march in the case of order<1 dry

p.k = 2;
c0 = 0.01;
% der = 1e-30;
der = 1e-8;
start = 1;


%% ODE's for order < 1

function xdot = ivp(t, x, order, p)
    xdot = zeros(2, 1);
    xdot(1) = x(2);

    if t == 0
        xdot(2) = p.k*x(1)^order;
    else
        xdot(2) = -p.q/t*x(2) + p.k*x(1)^order;
    endif
endfunction

%% Stopping Critierion for order > 2

function [stp, isterminal, direction] = g(t, x, order, p)
    stp = sqrt((order + 1)/2*x(1)^(order - 1)*p.k) * t/(p.q + 1) - p.thiele;
    isterminal = 1;
    direction = 0;
endfunction

%% Main program

no = length(ordervec);

for j = 1:no
    order = ordervec(j);

    if (order < 1)

        %% Stage 1
        tsteps = [0; logspace(-2,5,200)'];
        x0 = [c0; 0];
        ode_opts = odeset('AbsTol', 1e-18, 'RelTol', sqrt(eps));
        [tsolver,x] = ode15s(@(t, x) ivp(t, x, order, p), tsteps, x0, ode_opts);
        phi = sqrt((order + 1)/2.*x(:,1).^(order - 1)*p.k) .* tsteps/(p.q + 1);
        eta = x(:,2)./(p.k*(tsteps).*x(:,1).^order)*(p.q + 1);
        phi(1) = 0;
        eta(1) = 1.0;

        %% Stage 2
        tsteps = [linspace(start,start + 2,30)';
                  logspace(log10(start + 2), log10(40),20)'];
        x0 = [0; der];
        ode_opts = odeset('AbsTol', 1e-18, 'RelTol', sqrt(eps));
        [tsolver,x] = ode15s(@(t, x) ivp(t, x, order, p), tsteps, x0, ode_opts);
        phidry = sqrt((order + 1)/2.*x(:,1).^(order-1)*p.k) .* ...
                    (tsteps + start)/(p.q + 1);
        etadry = x(:,2)./(p.k*(tsteps + start).*x(:,1).^order)*(p.q + 1);
        tab = [phi, eta;
               flipud(phidry(2:end)), flipud(etadry(2:end))];

    elseif (order == 1)
        phi = logspace(-2,2,250)';

        if p.q == 0
            eta = tanh(phi)./phi;
        elseif p.q == 1
            eta = besseli(2*phi,1)./phi./besseli(2*phi,1);
        elseif p.q == 2
            eta = (1./phi).*((1./tanh(3*phi)) - (1./(3*phi)));
        endif

        tab = [phi eta];
        %    save ('-ascii', sprintf ('etaordern_large_%d.dat', j), 'tab');

    elseif order > 1
        x_0 = [c0; 0];
        y = p.k*c0^order;
        xdot_0 = [0; y];
        t_out = [0; logspace(-1,10,2000)'];

        ode_opts = odeset('Events', @(t,x) g(t, x, order, p),...
                          'AbsTol', sqrt(eps), 'RelTol', sqrt(eps));
        [t,x] = ode15s(@(t, x) ivp(t, x, order, p), t_out, x_0, ode_opts);
        phi = sqrt((order + 1)/2.*x(:,1).^(order - 1)*p.k) .* t./(p.q + 1);
        eta = x(:,2)./(p.k*t.*x(:,1).^order)*(p.q + 1);
        tab = [phi(2:end,:) eta(2:end,:)];
        %    save ('-ascii', sprintf ('etaordern_large_%d.dat', j), 'tab');
    endif

    table(j) = {tab};

    % if (~ strcmp (getenv ('OMIT_PLOTS'), 'true')) %% PLOTTING
    % hold ('on');
    % loglog (tab(:,1), tab(:,2));
    % axis([0.1, p.thiele, 1/p.thiele, 1])
    % %% TITLE
    % endif %% PLOTTING
endfor

%hold ('off');

save etaordern_large.dat table

if (~ strcmp (getenv ('OMIT_PLOTS'), 'true')) %% PLOTTING
    hold on;

    for i = 1:no
        loglog(table{i}(:,1), table{i}(:,2));
    endfor

    hold off;
    axis([0.1, p.thiele, 1/p.thiele, 1])
    %% TITLE
endif %% PLOTTING