%% Copyright (C) 2001, James B. Rawlings and John G. Ekerdt
%%
%% This program is free software; you can redistribute it and/or
%% modify it under the terms of the GNU General Public License as
%% published by the Free Software Foundation; either version 2, or (at
%% your option) any later version.
%%
%% This program is distributed in the hope that it will be useful, but
%% WITHOUT ANY WARRANTY; without even the implied warranty of
%% MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
%% General Public License for more details.
%%
%% You should have received a copy of the GNU General Public License
%% along with this program; see the file COPYING.  If not, write to
%% the Free Software Foundation, 59 Temple Place - Suite 330, Boston,
%% MA 02111-1307, USA.
%%
%% Revised 8/16/2018

a = 5;
b = 0.05;
c0 = 5;
conv = 0.95;
d  = sqrt((1 - 2*b)*(1 - 2*b) - 4*b*(b + 1));
w1 = (1-2*b + d)/(2*b);
w2 = (1-2*b - d)/(2*b);
x1  = sqrt(w1/a);
x2  = sqrt(w2/a);

function rate = rxrate(c, a, b)
    rate = c./(1+a*c.*c) + b*c;
endfunction

c = linspace(0.01,10,300)';
y = 1./rxrate(c, a, b);

%% find the outlet of the first PFR, c1, which satisfies r(c1) = 1/x2
function retval = solverate(c, a, b, fixrate)
    retval = fixrate - rxrate(c, a, b);
endfunction

fixrate = rxrate(x2, a, b);
x0 = c0;
[x, fval, info] = fsolve(@(c) solverate(c, a, b, fixrate), x0);
info;
c1 = x;

%% size the first PFR, theta1 is size such that c(theta1) = c1
function dtdc = pfr(c, t, a, b)
    dtdc = - 1/rxrate(c, a, b);
endfunction

t0 = 0;
ncs = 10;
cout = linspace(c0,c1,ncs);
opts = odeset('AbsTol', sqrt(eps), 'RelTol', sqrt(eps));
[tsolver, tout] = ode15s(@(c, t) pfr(c, t, a, b), cout, t0, opts);
theta1 = tout(ncs);

%% size the cstr, theta2= (c1-c2)/r(c2), c2 = x2
c2 = x2;
theta2 = (c1-c2)/rxrate(c2, a, b);

%% size the second PFR, c3, which satisfies c3 = (1-conv)*c0
c3 = (1 - conv)*c0;
t0 = 0;
ncs = 10;
cout = linspace(c2,c3,ncs);
opts = odeset('AbsTol', sqrt(eps), 'RelTol', sqrt(eps));
[tsolver, tout] = ode15s(@(c, t) pfr(c, t, a, b), cout, t0, opts);
theta3 = tout(ncs);

% c0; c1; c2; c3;
% theta1; theta2; theta3;
%%
%% optimal reactor is PFR --->    CSTR --->  PFR
%%                    theta1      theta2     theta3

%%compute the rtd
npts = 100;
thetamin = theta1 + theta3;
thetamax =  thetamin + 5*theta2;
theta = linspace(thetamin,thetamax,npts)';
p = 1/theta2*exp(-(theta - (theta1 + theta3))/theta2);
table = [0 0; thetamin 0; [theta p]];

save -ascii leven_rtd.dat table;

if (~ strcmp (getenv ('OMIT_PLOTS'), 'true')) %% PLOTTING
    plot (table(:,1), table(:,2));
    %% TITLE
endif %% PLOTTING