%% Copyright (C) 2001, James B. Rawlings and John G. Ekerdt
%%
%% This program is free software; you can redistribute it and/or
%% modify it under the terms of the GNU General Public License as
%% published by the Free Software Foundation; either version 2, or (at
%% your option) any later version.
%%
%% This program is distributed in the hope that it will be useful, but
%% WITHOUT ANY WARRANTY; without even the implied warranty of
%% MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
%% General Public License for more details.
%%
%% You should have received a copy of the GNU General Public License
%% along with this program; see the file COPYING.  If not, write to
%% the Free Software Foundation, 59 Temple Place - Suite 330, Boston,
%% MA 02111-1307, USA.
%%
%% Revised 8/29/2018

%%
%% vrdiclo                      initial moles of dichloro
%% k1k2ratio                    k1/k2
%% vro	                        initial reactor volume  cu dm
%% teef	                        TEA feed concentration kgmol/cu dm
%% teaden                       TEA density kg/cu dm
%% tflow, flowrate              measured flowrate at time tflow
%%                              read from "flow.dat" containing time
%%                              (min) and flowrate (kg/min) converted to
%%                              (cu dm/min)
%% tlc, lcmeas                  lc measurement at time tlc (min)
%%
%% optimal values

% This m-file loads data file 'flow.dat'.
% This m-file loads data file 'lc.dat'.
p.vrdiclo   =  2.3497;
p.k1k2ratio =  2;
p.vro       = 2370;
p.teaf      = 0.00721;
p.teaden    = 0.728;

flow = load('flow.dat');
lc = load('lc.dat');
p.tflow  = [0.; flow(:,1)];
p.flowrate = [0.; flow(:,2)./p.teaden];
tlc    = lc(:,1);
lcmeas = lc(:,2);

ntimes = 80;
tlin = linspace(0, p.tflow(end), ntimes)';
p.timeout = sort(unique([tlin; p.tflow; tlc]));

%% I can't think of a way to do this without a loop.
idx = zeros(size(tlc));

for i = 1:length (tlc)
    %% This assignment will fail if find returns more than one element,
    %% but that's OK, because each element of tlc should only appear once
    %% in timeout.
    idx(i) = find (p.timeout == tlc(i));
endfor

%% sanity check
if (any (p.timeout (idx) ~= tlc))
    error ('extracting index vector for tlc');
endif

function xdot = res_reduced(time, x, p)
    %%	Passed variable                  Local equivalent
    %%
    %%	x(1)                             reactor contents volume
    %%	x(2)                             extent of second reaction
    %%
    vrdiclo = p.theta(1);
    k1k2ratio = p.theta(2);
    vr = x(1);

    if ( vr <= 0.0 )
        error ('(res_reduced).... vr is zero, time= %d',time);
    endif

    eps2   = x(2);
    badded = (vr - p.vro)*p.teaf;
    eps1   = badded - eps2;

    %% look up flowrate at current time
    [index, is_switching_time] = find_time_index(p.tflow, time);
    fteaf   = p.flowrate(index) * p.teaf;

    %%	calculate xdots
    xdot(1) = p.flowrate(index);

    if (badded == 0.)
        xdot(2) = 0.;
    else
        xdot(2) = fteaf/( 1 + p.k1k2ratio*(vrdiclo - (badded - eps2) )/...
                    (badded - 2*eps2) );
    endif
    xdot = xdot(:);
end%function

x0 = [p.vro; 0];
opts = odeset('AbsTol', sqrt(eps));
ks = [0.5; 1; 2; 8; 32];
nks = length(ks);

for i = 1:nks

    p.k1k2ratio = ks(i);
    p.theta(1) = p.vrdiclo;
    p.theta(2) = p.k1k2ratio;
    [dummy, x] = ode15s(@(time, x) res_reduced(time, x, p), p.timeout, x0, opts);

    %% express the other states in terms of vr and eps2
    vr   = x(:,1);
    eps2 = x(:,2);
    badded = (vr - p.vro)*p.teaf;
    eps1 = badded - eps2;

    ca = (p.vrdiclo - eps1)./vr;
    cb = badded - (eps1 + eps2);  % sanity check, should be zero
    cc = (eps1 - eps2)./vr;
    ccd = eps2./vr;
    lcpred(:,i) = cc./(cc + 2*ccd);

    %% avoid 0/0 at time = 0 and next few times if flowrate is zero and
    %% therefore x doesn't change;
    firstposflow = min(find(p.flowrate > 0));
    lcpred(1:firstposflow,i) = 1;

endfor

table = [p.timeout lcpred];
data = struct('lc', lc, 'table', table);
gnuplotsave('lc_reduced.dat', data);

if (~ strcmp (getenv ('OMIT_PLOTS'), 'true')) %% PLOTTING
    plot (table(:,1), table(:,2:6), lc(:,1), lc(:,2), '+');
    %% TITLE
endif %% PLOTTING