function failures = runall(doplots)
% failures = runall([showplots='off'])
%
% Runs all of the figure scripts and returns a cell array of errors for any
% scripts that failed.
%
% If showplots is true, plots will be displayed as they are created (note that
% they will steal focus, which can be annoying). Otherwise, they will remain
% hidden until the script finishes (at which point they will also steal focus).
% Finally, if showplots is the string 'off', then they will never be shown.
narginchk(0, 1);
clearplots = false();
if nargin() < 1
    doplots = 'off';
end
if isequal(doplots, 'off')
    doplots = false();
    clearplots = true();
end

% Make sure user has the required packages.
book_setup('error');

% Set some local printing options if octave.
if isOctave()
    more('off');
    page_output_immediately(true(), 'local');
    page_screen_output(false(), 'local');
end

% Start with a dummy NLP to display the IPOPT splash message.
x = casadi.SX.sym('x');
nlp = struct('x', x, 'f', x.^2);
ipopt = casadi.nlpsol('ipopt', 'ipopt', nlp, ...
                      struct('print_time', false(), ...
                             'ipopt', struct('print_level', 0)));
ipopt('x0', 0);

% Switch off plotting.
DefaultFigureVisible = get(0, 'DefaultFigureVisible');
if ~doplots
    set(0, 'DefaultFigureVisible', 'off'); % Don't show figures.
end
initialfigures = get(0, 'children');
getnewplots = @() setdiff(get(0, 'children'), initialfigures);

% Now actually run the examples.
fprintf('\nRunning book scripts.\n\n');
bookfigures = {
    % Only runs unique figure scripts.
    'ch2/fig-2.2';
    'ch2/fig-2.3';
    'ch3/fig-3.3';
    'ch3/fig-3.4';
    'ch3/fig-3.6';
    'ch4/fig-4.3';
    'ch4/fig-4.4';
    'ch4/fig-4.5';
    'ch4/fig-4.6';
    'ch4/fig-4.7';
    'ch4/fig-4.8';
    'ch4/fig-4.9';
    'ch4/fig-4.10';
    'ch4/fig-4.11';
    'ch4/fig-4.12';
    'ch4/fig-4.14';
    'ch4/fig-4.15';
    'ch4/fig-4.20';
    'ch4/fig-4.22';
    'ch4/fig-4.23';
    'ch4/fig-4.27';
    'ch4/fig-4.29';
    'ch4/fig-4.30';
    'ch4/fig-4.31';
    'ch4/fig-4.32';
    'ch4/fig-4.33';
    'ch4/fig-4.34';
    'ch4/fig-4.35';
    'ch4/fig-4.37';
    'ch5/fig-5.1';
    'ch5/fig-5.2';
    'ch5/fig-5.3';
    'ch5/fig-5.4';
    'ch5/fig-5.5';
    'ch5/fig-5.6';
    'ch5/fig-5.7';
    'ch5/fig-5.8';
    'ch5/fig-5.9';
    'ch5/fig-5.10';
    'ch5/fig-5.11';
    'ch5/fig-5.12';
    'ch5/fig-5.13';
    'ch5/fig-5.14';
    'ch5/fig-5.15';
    'ch5/fig-5.18';
    'ch5/fig-5.19';
    'ch5/fig-5.20';
    'ch5/fig-5.23';
    'ch5/fig-5.24';
    'ch6/fig-6.4';
    'ch6/fig-6.5';
    'ch6/fig-6.7';
    'ch6/fig-6.9';
    'ch6/fig-6.10';
    'ch6/fig-6.13';
    'ch6/fig-6.18';
    'ch6/fig-6.22';
    'ch6/fig-6.24';
    'ch6/fig-6.25';
    'ch6/fig-6.26';
    'ch6/fig-6.27';
    'ch6/fig-6.28';
    'ch6/fig-6.29';
    'ch6/fig-6.30';
    'ch6/fig-6.35';
    'ch6/fig-6.38';
    'ch6/fig-6.39';
    'ch7/fig-7.3';
    'ch7/fig-7.4';
    'ch7/fig-7.5';
    'ch7/fig-7.6';
    'ch7/fig-7.8';
    'ch7/fig-7.9';
    'ch7/fig-7.10';
    'ch7/fig-7.11';
    'ch7/fig-7.13';
    'ch7/fig-7.16';
    'ch7/fig-7.17';
    'ch7/fig-7.18';
    'ch7/fig-7.19';
    'ch7/fig-7.20';
    'ch7/fig-7.22';
    'ch7/fig-7.26';
    'ch7/fig-7.27';
    'ch7/fig-7.29';
    'ch8/fig-8.5';
    'ch8/fig-8.8';
    'ch8/fig-8.9';
    'ch8/fig-8.10';
    'ch8/fig-8.11';
    'ch8/fig-8.12';
    'ch8/fig-8.14';
    'ch8/fig-8.19';
    'ch8/fig-8.22';
    'ch8/fig-8.29';
    'ch8/fig-8.30';
    'ch8/fig-8.31';
    'ch8/fig-8.33';
    'ch8/fig-8.36';
    'ch9/fig-9.4';
    'ch9/fig-9.5';
    'ch9/fig-9.7';
    'ch9/fig-9.8';
    'ch9/fig-9.9';
    'ch9/fig-9.10';
    'ch9/fig-9.11';
    'ch9/fig-9.12';
    'ch9/fig-9.13';
    'ch9/fig-9.14';
    'ch9/fig-9.15';
    'ch9/fig-9.16';
    'ch9/fig-9.18';
    'ch9/fig-9.19';
    'ch9/fig-9.20';
    'ch9/fig-9.21';
    'ch9/fig-9.22';
    'ch9/fig-9.23';
    'ch9/fig-9.31';
    'ch9/fig-9.32';
    'ch9/fig-9.35';
    'ch9/fig-9.36';
    'ch10/fig-10.12';
    'ch10/fig-10.14';
    'ch10/fig-10.15';
    'appendix/fig-A.1';
    'appendix/fig-A.3';
    'appendix/fig-A.5';
    'appendix/fig-A.6';
    'appendix/fig-A.7';
    'appendix/fig-A.10';
};
failures = cell(0, 1);
for i = 1:length(bookfigures)
    fig = bookfigures{i};
    fprintf('* %s ... ', fig);
    [err, figtime] = runmain(fig);
    if isempty(err)
        fprintf('success');
    else
        fprintf('FAILURE');
        failures{length(failures) + 1} = struct('figure', fig, 'err', err);
    end
    fprintf(' (%.4g s)\n', figtime);
    if clearplots
        close(getnewplots()); % Close figures to save memory.
    end
end

% Make figures show up.
if ~doplots && ~clearplots
    newplots = getnewplots();
    for i = 1:length(newplots)
        set(newplots(i), 'Visible', 'on');
    end
end

% Switch plots back on.
set(0, 'DefaultFigureVisible', DefaultFigureVisible);

% Display failures if output not requested.
if nargout() == 0
    for i = 1:length(failures)
        f = failures{i};
        fprintf('*** %s ***\n', f.figure);
        disp(f.err);
        
        % Print the stack trace, but skip the last two (they will always be
        % runall, runmain, and runscript_octave/matlab.
        for j = 1:(length(f.err.stack) - 3)
            disp(f.err.stack(j));
        end
    end
end

end%function

function [err, figtime] = runmain(fig)
    % [err, figtime] = runmain(fig)
    %
    % Changes to the directory of of the given figure and runs main.m, returning
    % any error message. Afterwards, changes back to the original directory.
    olddir = cd(fig);
    dircleanup = onCleanup(@() cd(olddir));
    oldpath = path();
    pathcleanup = onCleanup(@() path(oldpath));
    
    figure(); % Avoids issues with reusing old figures.
    figtime = tic();
    if isOctave()
        err = runscript_octave('main');
    else
        err = runscript_matlab('main');
    end
    figtime = toc(figtime);
end%function

function err = runscript_matlab(script)
    % err = runscript_matlab(script)
    %
    % Attempts to run a script without displaying anything and returns any error
    % message.
    try
        evalc(script);
        err = [];
    catch err
        % Pass.
    end
end%function

function err = runscript_octave(script)
    % err = runscript_octave(script)
    %
    % Attempts to run a script and returns any error message.
    %
    % Most output should be suppressed, although we can't hit everything.
    clear('-x', 'script'); % Clear any lingering function definitions.
    try
        evalc(script);
        err = [];
    catch err
        % Pass.
    end
end%function

function noop(varargin)
    % Does nothing.
end%function

